const express = require("express");
const jwt = require("jsonwebtoken");
const bcrypt = require("bcrypt");
const router = express.Router();
const nodemailer = require("nodemailer");
const AdminModel = require("../models/Admin.model");
const { isAdminAuthenticated } = require("../middlewares/authenticate");

// ------------------------------
// Admin Login Route (Debug + Fixed)
// ------------------------------
router.post("/login", async (req, res) => {
  console.log("📩 Full Request Body:", req.body);
  console.log("📧 Email from body:", req.body.email);
  console.log("🔑 Password from body:", req.body.password);

  const { email, password } = req.body;

  try {
    const admin = await AdminModel.findOne({ email });
    console.log("🟣 Found admin:", admin ? admin.email : "Not found");

    if (!admin) {
      console.log("❌ No admin found for this email");
      return res.status(401).send({ message: "Wrong credentials (no admin)" });
    }

    console.log("🔒 Hashed Password in DB:", admin.password);

    const isMatch = await bcrypt.compare(password, admin.password);
    console.log("🟠 Password Match Result:", isMatch);

    if (!isMatch) {
      console.log("❌ Password did not match");
      return res.status(401).send({ message: "Wrong credentials (password mismatch)" });
    }

    if (admin.access === "false") {
      console.log("⛔ Access denied for this admin");
      return res.send({ message: "Access Denied" });
    }

    const token = jwt.sign(
      { email: admin.email, name: admin.name },
      process.env.SECRET_KEY,
      { expiresIn: "7d" }
    );

    console.log("✅ Login successful for:", admin.email);

    res.status(200).send({
      message: "Login Successful",
      user: admin,
      token,
    });
  } catch (error) {
    console.error("❌ Error in login:", error.message);
    res.status(500).send({ message: "Server Error", error: error.message });
  }
});

// ------------------------------
// Other Routes (optional)
// ------------------------------
router.get("/all", async (req, res) => {
  try {
    const admins = await AdminModel.find();
    res.status(200).send({ message: "All admins fetched successfully", admins });
  } catch (error) {
    res.status(500).send({ message: "Error fetching admins", error: error.message });
  }
});

router.post("/register", isAdminAuthenticated, async (req, res) => {
  try {
    const { name, email, password } = req.body;
    const existing = await AdminModel.findOne({ email });

    if (existing) {
      return res.status(400).send({ message: "Admin already exists" });
    }

    const hashed = await bcrypt.hash(password, 5);
    const newAdmin = new AdminModel({
      name,
      email,
      password: hashed,
      userType: "Admin",
    });

    await newAdmin.save();
    res.status(201).send({ message: "Admin Registered Successfully", admin: newAdmin });
  } catch (error) {
    res.status(500).send({ message: "Error registering admin", error: error.message });
  }
});

router.patch("/:adminId", isAdminAuthenticated, async (req, res) => {
  try {
    const { adminId } = req.params;
    const payload = req.body;
    await AdminModel.findByIdAndUpdate(adminId, payload);
    const updatedAdmin = await AdminModel.findById(adminId);
    res.status(200).send({ message: "Admin Updated Successfully", admin: updatedAdmin });
  } catch (error) {
    res.status(500).send({ message: "Error updating admin", error: error.message });
  }
});

router.delete("/:adminId", isAdminAuthenticated, async (req, res) => {
  try {
    const { adminId } = req.params;
    await AdminModel.findByIdAndDelete(adminId);
    res.status(200).send({ message: "Admin Deleted Successfully" });
  } catch (error) {
    res.status(500).send({ message: "Error deleting admin", error: error.message });
  }
});

module.exports = router;
